<?php
/**
 * Classe de administração do plugin UTM TRK PRO
 */
class UTM_TRK_Pro_Admin {
    
    // Instância da classe de configurações
    private $settings;
    
    /**
     * Construtor
     */
    public function __construct($settings) {
        $this->settings = $settings;
        
        // Registrar o menu de administração
        add_action('admin_menu', array($this, 'register_admin_menu'));
        
        // Registrar as configurações
        add_action('admin_init', array($this, 'register_settings'));
        
        // Adicionar scripts e estilos de administração
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Registra o menu de administração
     */
    public function register_admin_menu() {
        add_menu_page(
            esc_html__('UTM TRK PRO', 'utm-trk-pro'),
            esc_html__('UTM TRK PRO', 'utm-trk-pro'),
            'manage_options',
            'utm-trk-pro',
            array($this, 'render_dashboard_page'),
            'dashicons-chart-line',
            85
        );
        
        add_submenu_page(
            'utm-trk-pro',
            esc_html__('Dashboard', 'utm-trk-pro'),
            esc_html__('Dashboard', 'utm-trk-pro'),
            'manage_options',
            'utm-trk-pro',
            array($this, 'render_dashboard_page')
        );
        
        add_submenu_page(
            'utm-trk-pro',
            esc_html__('Configurações', 'utm-trk-pro'),
            esc_html__('Configurações', 'utm-trk-pro'),
            'manage_options',
            'utm-trk-pro-settings',
            array($this, 'render_settings_page')
        );
        
        add_submenu_page(
            'utm-trk-pro',
            esc_html__('Gerador de URL', 'utm-trk-pro'),
            esc_html__('Gerador de URL', 'utm-trk-pro'),
            'manage_options',
            'utm-trk-pro-generator',
            array($this, 'render_generator_page')
        );
    }
    
    /**
     * Registra as configurações
     */
    public function register_settings() {
        register_setting(
            'utm_trk_pro_settings', 
            'utm_trk_pro_options',
            array(
                'sanitize_callback' => array($this->settings, 'sanitize_options')
            )
        );
        
        // Seção Principal
        add_settings_section(
            'utm_trk_pro_main_section',
            esc_html__('Configurações Gerais', 'utm-trk-pro'),
            array($this, 'render_main_section_info'),
            'utm-trk-pro-settings'
        );
        
        // Campo para habilitar/desabilitar o rastreamento
        add_settings_field(
            'enable_tracking',
            esc_html__('Habilitar Rastreamento', 'utm-trk-pro'),
            array($this, 'render_checkbox_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_main_section',
            array(
                'id' => 'enable_tracking',
                'label' => esc_html__('Ativar o rastreamento de parâmetros UTM', 'utm-trk-pro'),
            )
        );
        
        // Campo para habilitar/desabilitar o rastreamento de links externos
        add_settings_field(
            'track_external',
            esc_html__('Rastrear Links Externos', 'utm-trk-pro'),
            array($this, 'render_checkbox_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_main_section',
            array(
                'id' => 'track_external',
                'label' => esc_html__('Adicionar parâmetros UTM a links externos', 'utm-trk-pro'),
            )
        );
        
        // Campo para habilitar/desabilitar o rastreamento de links internos
        add_settings_field(
            'track_internal',
            esc_html__('Rastrear Links Internos', 'utm-trk-pro'),
            array($this, 'render_checkbox_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_main_section',
            array(
                'id' => 'track_internal',
                'label' => esc_html__('Adicionar parâmetros UTM a links internos', 'utm-trk-pro'),
            )
        );
        
        // Campo para domínios excluídos
        add_settings_field(
            'excluded_domains',
            esc_html__('Domínios Excluídos', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_main_section',
            array(
                'id' => 'excluded_domains',
                'label' => esc_html__('Lista de domínios que não devem receber parâmetros UTM (separados por vírgula)', 'utm-trk-pro'),
                'placeholder' => 'exemplo.com, outrosite.com.br'
            )
        );
        
        // Seção de Cookies
        add_settings_section(
            'utm_trk_pro_cookies_section',
            esc_html__('Configurações de Cookies', 'utm-trk-pro'),
            array($this, 'render_cookies_section_info'),
            'utm-trk-pro-settings'
        );
        
        // Campo para duração dos cookies
        add_settings_field(
            'cookie_duration',
            esc_html__('Duração dos Cookies', 'utm-trk-pro'),
            array($this, 'render_number_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_cookies_section',
            array(
                'id' => 'cookie_duration',
                'label' => esc_html__('Dias', 'utm-trk-pro'),
                'min' => 1,
                'max' => 365,
                'step' => 1
            )
        );
        
        // Seção de Parâmetros UTM Padrão
        add_settings_section(
            'utm_trk_pro_default_utm_section',
            esc_html__('Parâmetros UTM Padrão', 'utm-trk-pro'),
            array($this, 'render_default_utm_section_info'),
            'utm-trk-pro-settings'
        );
        
        // Campo para UTM source padrão
        add_settings_field(
            'default_utm_source',
            esc_html__('UTM Source Padrão', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_default_utm_section',
            array(
                'id' => 'default_utm_source',
                'placeholder' => 'website'
            )
        );
        
        // Campo para UTM medium padrão
        add_settings_field(
            'default_utm_medium',
            esc_html__('UTM Medium Padrão', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_default_utm_section',
            array(
                'id' => 'default_utm_medium',
                'placeholder' => 'organic'
            )
        );
        
        // Campo para UTM campaign padrão
        add_settings_field(
            'default_utm_campaign',
            esc_html__('UTM Campaign Padrão', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_default_utm_section',
            array(
                'id' => 'default_utm_campaign',
                'placeholder' => 'website_traffic'
            )
        );
        
        // Campo para UTM content padrão
        add_settings_field(
            'default_utm_content',
            esc_html__('UTM Content Padrão', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_default_utm_section',
            array(
                'id' => 'default_utm_content',
                'placeholder' => 'navlink'
            )
        );
        
        // Campo para UTM term padrão
        add_settings_field(
            'default_utm_term',
            esc_html__('UTM Term Padrão', 'utm-trk-pro'),
            array($this, 'render_text_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_default_utm_section',
            array(
                'id' => 'default_utm_term',
                'placeholder' => 'marketing'
            )
        );
        
        // Seção de Integrações
        add_settings_section(
            'utm_trk_pro_integrations_section',
            esc_html__('Integrações', 'utm-trk-pro'),
            array($this, 'render_integrations_section_info'),
            'utm-trk-pro-settings'
        );
        
        // Campo para habilitar/desabilitar a integração com o Google Analytics
        add_settings_field(
            'enable_analytics',
            esc_html__('Google Analytics', 'utm-trk-pro'),
            array($this, 'render_checkbox_field'),
            'utm-trk-pro-settings',
            'utm_trk_pro_integrations_section',
            array(
                'id' => 'enable_analytics',
                'label' => esc_html__('Habilitar integração com o Google Analytics', 'utm-trk-pro'),
            )
        );
    }
    
    /**
     * Carrega scripts e estilos de administração
     */
    public function enqueue_admin_scripts($hook) {
        // Carregar apenas nas páginas do plugin
        if (strpos($hook, 'utm-trk-pro') === false) {
            return;
        }
        
        // Registrar e carregar o CSS
        wp_register_style(
            'utm-trk-pro-admin',
            UTM_TRK_PRO_URL . 'assets/css/admin.css',
            array(),
            UTM_TRK_PRO_VERSION
        );
        wp_enqueue_style('utm-trk-pro-admin');
        
        // Registrar e carregar o JavaScript
        wp_register_script(
            'utm-trk-pro-admin',
            UTM_TRK_PRO_URL . 'assets/js/admin.js',
            array('jquery'),
            UTM_TRK_PRO_VERSION,
            true
        );
        
        // Passar variáveis para o script
        wp_localize_script(
            'utm-trk-pro-admin',
            'utmTrkProAdmin',
            array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('utm-trk-pro-nonce')
            )
        );
        
        wp_enqueue_script('utm-trk-pro-admin');
    }
    
    /**
     * Renderiza informações da seção principal
     */
    public function render_main_section_info() {
        echo '<p>' . esc_html__('Configure o comportamento geral do rastreamento de UTM.', 'utm-trk-pro') . '</p>';
    }
    
    /**
     * Renderiza informações da seção de cookies
     */
    public function render_cookies_section_info() {
        echo '<p>' . esc_html__('Configure como os parâmetros UTM são armazenados em cookies.', 'utm-trk-pro') . '</p>';
    }
    
    /**
     * Renderiza informações da seção de UTM padrão
     */
    public function render_default_utm_section_info() {
        echo '<p>' . esc_html__('Defina parâmetros UTM padrão para serem usados quando não houver parâmetros na URL.', 'utm-trk-pro') . '</p>';
    }
    
    /**
     * Renderiza informações da seção de integrações
     */
    public function render_integrations_section_info() {
        echo '<p>' . esc_html__('Configure integrações com outras ferramentas de analytics.', 'utm-trk-pro') . '</p>';
    }
    
    /**
     * Renderiza campo de checkbox
     */
    public function render_checkbox_field($args) {
        $id = $args['id'];
        $label = isset($args['label']) ? $args['label'] : '';
        $value = $this->settings->get_option($id, 'no');
        
        echo '<label for="' . esc_attr($id) . '">';
        echo '<input type="checkbox" id="' . esc_attr($id) . '" name="utm_trk_pro_options[' . esc_attr($id) . ']" value="yes" ' . checked($value, 'yes', false) . '>';
        echo esc_html($label) . '</label>';
    }
    
    /**
     * Renderiza campo de texto
     */
    public function render_text_field($args) {
        $id = $args['id'];
        $label = isset($args['label']) ? $args['label'] : '';
        $placeholder = isset($args['placeholder']) ? $args['placeholder'] : '';
        $value = $this->settings->get_option($id, '');
        
        echo '<input type="text" id="' . esc_attr($id) . '" name="utm_trk_pro_options[' . esc_attr($id) . ']" value="' . esc_attr($value) . '" placeholder="' . esc_attr($placeholder) . '" class="regular-text">';
        
        if (!empty($label)) {
            echo '<p class="description">' . esc_html($label) . '</p>';
        }
    }
    
    /**
     * Renderiza campo numérico
     */
    public function render_number_field($args) {
        $id = $args['id'];
        $label = isset($args['label']) ? $args['label'] : '';
        $min = isset($args['min']) ? $args['min'] : '';
        $max = isset($args['max']) ? $args['max'] : '';
        $step = isset($args['step']) ? $args['step'] : '1';
        $value = $this->settings->get_option($id, '');
        
        echo '<input type="number" id="' . esc_attr($id) . '" name="utm_trk_pro_options[' . esc_attr($id) . ']" value="' . esc_attr($value) . '" min="' . esc_attr($min) . '" max="' . esc_attr($max) . '" step="' . esc_attr($step) . '" class="small-text">';
        
        if (!empty($label)) {
            echo ' ' . esc_html($label);
        }
    }
    
    /**
     * Renderiza a página do dashboard
     */
    public function render_dashboard_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="utm-trk-pro-dashboard">
                <div class="utm-trk-pro-card">
                    <h2><?php esc_html_e('Status do Rastreamento', 'utm-trk-pro'); ?></h2>
                    <?php if ($this->settings->is_tracking_enabled()): ?>
                        <div class="utm-trk-pro-status utm-trk-pro-status-active">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <p><?php esc_html_e('O rastreamento de UTM está ativo', 'utm-trk-pro'); ?></p>
                        </div>
                    <?php else: ?>
                        <div class="utm-trk-pro-status utm-trk-pro-status-inactive">
                            <span class="dashicons dashicons-no-alt"></span>
                            <p><?php esc_html_e('O rastreamento de UTM está desativado', 'utm-trk-pro'); ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <div class="utm-trk-pro-details">
                        <p>
                            <strong><?php esc_html_e('Links Internos:', 'utm-trk-pro'); ?></strong>
                            <?php echo $this->settings->is_internal_tracking_enabled() ? esc_html__('Ativado', 'utm-trk-pro') : esc_html__('Desativado', 'utm-trk-pro'); ?>
                        </p>
                        <p>
                            <strong><?php esc_html_e('Links Externos:', 'utm-trk-pro'); ?></strong>
                            <?php echo $this->settings->is_external_tracking_enabled() ? esc_html__('Ativado', 'utm-trk-pro') : esc_html__('Desativado', 'utm-trk-pro'); ?>
                        </p>
                        <p>
                            <strong><?php esc_html_e('Duração dos Cookies:', 'utm-trk-pro'); ?></strong>
                            <?php echo esc_html($this->settings->get_cookie_duration()); ?> <?php esc_html_e('dias', 'utm-trk-pro'); ?>
                        </p>
                    </div>
                    
                    <a href="<?php echo esc_url(admin_url('admin.php?page=utm-trk-pro-settings')); ?>" class="button button-primary">
                        <?php esc_html_e('Editar Configurações', 'utm-trk-pro'); ?>
                    </a>
                </div>
                
                <div class="utm-trk-pro-card">
                    <h2><?php esc_html_e('Gerador de URL com UTM', 'utm-trk-pro'); ?></h2>
                    <p><?php esc_html_e('Crie URLs com parâmetros UTM para suas campanhas de marketing.', 'utm-trk-pro'); ?></p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=utm-trk-pro-generator')); ?>" class="button button-primary">
                        <?php esc_html_e('Acessar Gerador de URL', 'utm-trk-pro'); ?>
                    </a>
                </div>
            </div>
            
            <div class="utm-trk-pro-help">
                <h2><?php esc_html_e('Ajuda e Suporte', 'utm-trk-pro'); ?></h2>
                <p><?php esc_html_e('Para obter ajuda com o UTM TRK PRO, confira nossa documentação completa:', 'utm-trk-pro'); ?></p>
                <a href="https://www.rafaelarantes.com.br/plugins/utm-trk-pro/docs" target="_blank" class="button">
                    <?php esc_html_e('Documentação', 'utm-trk-pro'); ?>
                </a>
                <a href="https://www.rafaelarantes.com.br/plugins/utm-trk-pro/support" target="_blank" class="button">
                    <?php esc_html_e('Suporte Técnico', 'utm-trk-pro'); ?>
                </a>
            </div>
        </div>
        <?php
    }
    
    /**
     * Renderiza a página de configurações
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('utm_trk_pro_settings');
                do_settings_sections('utm-trk-pro-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Renderiza a página do gerador de URL
     */
    public function render_generator_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Obter os parâmetros UTM padrão
        $default_utm_params = $this->settings->get_default_utm_parameters();
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="utm-trk-pro-generator">
                <div class="utm-trk-pro-card">
                    <h2><?php esc_html_e('Gerador de URL com UTM', 'utm-trk-pro'); ?></h2>
                    
                    <div class="utm-trk-pro-generator-form">
                        <div class="utm-trk-pro-form-field">
                            <label for="url"><?php esc_html_e('URL de Destino', 'utm-trk-pro'); ?></label>
                            <input type="url" id="url" name="url" placeholder="https://seu-site.com.br" required>
                        </div>
                        
                        <div class="utm-trk-pro-form-field">
                            <label for="utm_source"><?php esc_html_e('Origem (utm_source)', 'utm-trk-pro'); ?></label>
                            <input type="text" id="utm_source" name="utm_source" value="<?php echo esc_attr($default_utm_params['utm_source'] ?? ''); ?>" placeholder="facebook, google, newsletter, etc." required>
                            <p class="description"><?php esc_html_e('De onde vem o tráfego (ex: facebook, google, newsletter)', 'utm-trk-pro'); ?></p>
                        </div>
                        
                        <div class="utm-trk-pro-form-field">
                            <label for="utm_medium"><?php esc_html_e('Meio (utm_medium)', 'utm-trk-pro'); ?></label>
                            <input type="text" id="utm_medium" name="utm_medium" value="<?php echo esc_attr($default_utm_params['utm_medium'] ?? ''); ?>" placeholder="cpc, banner, email, etc." required>
                            <p class="description"><?php esc_html_e('Como o tráfego chega até você (ex: cpc, banner, email)', 'utm-trk-pro'); ?></p>
                        </div>
                        
                        <div class="utm-trk-pro-form-field">
                            <label for="utm_campaign"><?php esc_html_e('Campanha (utm_campaign)', 'utm-trk-pro'); ?></label>
                            <input type="text" id="utm_campaign" name="utm_campaign" value="<?php echo esc_attr($default_utm_params['utm_campaign'] ?? ''); ?>" placeholder="black_friday, lancamento, etc." required>
                            <p class="description"><?php esc_html_e('Nome da sua campanha (ex: black_friday, lancamento)', 'utm-trk-pro'); ?></p>
                        </div>
                        
                        <div class="utm-trk-pro-form-field">
                            <label for="utm_content"><?php esc_html_e('Conteúdo (utm_content)', 'utm-trk-pro'); ?></label>
                            <input type="text" id="utm_content" name="utm_content" value="<?php echo esc_attr($default_utm_params['utm_content'] ?? ''); ?>" placeholder="banner_topo, botao_azul, etc.">
                            <p class="description"><?php esc_html_e('O que foi clicado (ex: banner_topo, botao_azul)', 'utm-trk-pro'); ?></p>
                        </div>
                        
                        <div class="utm-trk-pro-form-field">
                            <label for="utm_term"><?php esc_html_e('Termo (utm_term)', 'utm-trk-pro'); ?></label>
                            <input type="text" id="utm_term" name="utm_term" value="<?php echo esc_attr($default_utm_params['utm_term'] ?? ''); ?>" placeholder="marketing+digital, seo, etc.">
                            <p class="description"><?php esc_html_e('Termo de busca pago (ex: marketing+digital, seo)', 'utm-trk-pro'); ?></p>
                        </div>
                        
                        <button type="button" id="generate-utm-url" class="button button-primary">
                            <?php esc_html_e('Gerar URL', 'utm-trk-pro'); ?>
                        </button>
                    </div>
                    
                    <div class="utm-trk-pro-generator-result" style="display: none;">
                        <h3><?php esc_html_e('URL Gerada', 'utm-trk-pro'); ?></h3>
                        <div class="utm-trk-pro-url-result">
                            <textarea id="generated-url" readonly rows="3"></textarea>
                            <button type="button" id="copy-url" class="button">
                                <span class="dashicons dashicons-clipboard"></span>
                                <?php esc_html_e('Copiar', 'utm-trk-pro'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}