<?php
/**
 * Classe de configurações do plugin UTM TRK PRO
 */
class UTM_TRK_Pro_Settings {
    
    // Opções do plugin
    private $options;
    
    /**
     * Construtor
     */
    public function __construct() {
        $this->options = get_option('utm_trk_pro_options', array());
    }
    
    /**
     * Obtém uma opção específica
     */
    public function get_option($key, $default = '') {
        return isset($this->options[$key]) ? $this->options[$key] : $default;
    }
    
    /**
     * Obtém todas as opções
     */
    public function get_all_options() {
        return $this->options;
    }
    
    /**
     * Atualiza uma opção
     */
    public function update_option($key, $value) {
        $this->options[$key] = $value;
        return update_option('utm_trk_pro_options', $this->options);
    }
    
    /**
     * Atualiza todas as opções
     */
    public function update_all_options($options) {
        $this->options = $options;
        return update_option('utm_trk_pro_options', $this->options);
    }
    
    /**
     * Verifica se o tracking está habilitado
     */
    public function is_tracking_enabled() {
        return $this->get_option('enable_tracking', 'yes') === 'yes';
    }
    
    /**
     * Verifica se o tracking de links externos está habilitado
     */
    public function is_external_tracking_enabled() {
        return $this->get_option('track_external', 'yes') === 'yes';
    }
    
    /**
     * Verifica se o tracking de links internos está habilitado
     */
    public function is_internal_tracking_enabled() {
        return $this->get_option('track_internal', 'yes') === 'yes';
    }
    
    /**
     * Obtém a lista de domínios excluídos
     */
    public function get_excluded_domains() {
        $domains = $this->get_option('excluded_domains', '');
        if (empty($domains)) {
            return array();
        }
        
        // Converter a string de domínios em um array, remover espaços em branco
        $domains_array = array_map('trim', explode(',', $domains));
        return array_filter($domains_array);
    }
    
    /**
     * Obtém a duração dos cookies em dias
     */
    public function get_cookie_duration() {
        return (int) $this->get_option('cookie_duration', 30);
    }
    
    /**
     * Obtém os parâmetros UTM padrão
     */
    public function get_default_utm_parameters() {
        $default_params = array();
        
        $source = $this->get_option('default_utm_source', '');
        if (!empty($source)) {
            $default_params['utm_source'] = $source;
        }
        
        $medium = $this->get_option('default_utm_medium', '');
        if (!empty($medium)) {
            $default_params['utm_medium'] = $medium;
        }
        
        $campaign = $this->get_option('default_utm_campaign', '');
        if (!empty($campaign)) {
            $default_params['utm_campaign'] = $campaign;
        }
        
        $content = $this->get_option('default_utm_content', '');
        if (!empty($content)) {
            $default_params['utm_content'] = $content;
        }
        
        $term = $this->get_option('default_utm_term', '');
        if (!empty($term)) {
            $default_params['utm_term'] = $term;
        }
        
        return $default_params;
    }
    
    /**
     * Verifica se a integração com o Google Analytics está habilitada
     */
    public function is_analytics_enabled() {
        return $this->get_option('enable_analytics', 'no') === 'yes';
    }
    
    /**
     * Sanitiza as opções antes de salvar
     */
    public function sanitize_options($input) {
        $sanitized = array();
        
        // Checkboxes
        $sanitized['enable_tracking'] = isset($input['enable_tracking']) ? 'yes' : 'no';
        $sanitized['track_external'] = isset($input['track_external']) ? 'yes' : 'no';
        $sanitized['track_internal'] = isset($input['track_internal']) ? 'yes' : 'no';
        $sanitized['enable_analytics'] = isset($input['enable_analytics']) ? 'yes' : 'no';
        
        // Campos de texto
        $sanitized['excluded_domains'] = sanitize_text_field($input['excluded_domains']);
        $sanitized['cookie_duration'] = absint($input['cookie_duration']);
        
        // Campos UTM padrão
        $sanitized['default_utm_source'] = sanitize_text_field($input['default_utm_source']);
        $sanitized['default_utm_medium'] = sanitize_text_field($input['default_utm_medium']);
        $sanitized['default_utm_campaign'] = sanitize_text_field($input['default_utm_campaign']);
        $sanitized['default_utm_content'] = sanitize_text_field($input['default_utm_content']);
        $sanitized['default_utm_term'] = sanitize_text_field($input['default_utm_term']);
        
        return $sanitized;
    }
}
