<?php
/**
 * Classe de rastreamento do plugin UTM TRK PRO
 */
class UTM_TRK_Pro_Tracking {
    
    // Instância da classe de configurações
    private $settings;
    
    /**
     * Construtor
     */
    public function __construct($settings) {
        $this->settings = $settings;
        
        // Adicionar o script de rastreamento
        add_action('wp_footer', array($this, 'add_tracking_script'), 99);
        
        // Adicionar script de cookies
        add_action('wp_enqueue_scripts', array($this, 'enqueue_js_cookie'));
        
        // Registrar endpoint de AJAX para estatísticas
        add_action('wp_ajax_utm_trk_pro_track_click', array($this, 'track_click'));
        add_action('wp_ajax_nopriv_utm_trk_pro_track_click', array($this, 'track_click'));
    }
    
    /**
     * Carrega a biblioteca js-cookie
     */
    public function enqueue_js_cookie() {
        // Não carregar se o rastreamento estiver desativado
        if (!$this->settings->is_tracking_enabled()) {
            return;
        }
        
        wp_enqueue_script(
            'js-cookie',
            UTM_TRK_PRO_URL . 'assets/js/js.cookie.min.js',
            array(),
            '3.0.1',
            true
        );
    }
    
    /**
     * Adiciona o script de rastreamento
     */
    public function add_tracking_script() {
        // Não adicionar o script se o rastreamento estiver desativado
        if (!$this->settings->is_tracking_enabled()) {
            return;
        }
        
        // Obter configurações
        $track_internal = $this->settings->is_internal_tracking_enabled() ? 'true' : 'false';
        $track_external = $this->settings->is_external_tracking_enabled() ? 'true' : 'false';
        $excluded_domains = json_encode($this->settings->get_excluded_domains());
        $cookie_duration = $this->settings->get_cookie_duration();
        $enable_analytics = $this->settings->is_analytics_enabled() ? 'true' : 'false';
        $default_utm_params = json_encode($this->settings->get_default_utm_parameters());
        
        // Recuperar a URL base do site
        $site_url = parse_url(get_site_url(), PHP_URL_HOST);
        
        ?>
        <script type="text/javascript">
        document.addEventListener('DOMContentLoaded', function() {
            // Verifique se o js.cookie está disponível
            if (typeof Cookies === 'undefined') {
                console.error('UTM TRK PRO: js-cookie não está carregado corretamente');
                return;
            }
            
            // Configurações do plugin
            var utmTrkConfig = {
                trackInternal: <?php echo $track_internal; ?>,
                trackExternal: <?php echo $track_external; ?>,
                excludedDomains: <?php echo $excluded_domains; ?>,
                cookieDuration: <?php echo esc_js($cookie_duration); ?>,
                enableAnalytics: <?php echo $enable_analytics; ?>,
                defaultUtmParams: <?php echo $default_utm_params; ?>,
                siteHost: '<?php echo esc_js($site_url); ?>',
                ajaxUrl: '<?php echo esc_js(admin_url('admin-ajax.php')); ?>',
                nonce: '<?php echo esc_js(wp_create_nonce('utm-trk-pro-nonce')); ?>'
            };
            
            // Função para obter os parâmetros UTM da URL atual
            function getUTMParameters() {
                var params = {};
                var queryString = window.location.search.substring(1);
                
                if (queryString) {
                    var pairs = queryString.split('&');
                    for (var i = 0; i < pairs.length; i++) {
                        var pair = pairs[i].split('=');
                        var name = decodeURIComponent(pair[0]);
                        
                        // Verificar se é um parâmetro UTM
                        if (name.indexOf('utm_') === 0) {
                            params[name] = decodeURIComponent(pair[1] || '');
                            
                            // Armazenar o parâmetro em um cookie
                            Cookies.set(name, params[name], { expires: utmTrkConfig.cookieDuration });
                        }
                    }
                }
                
                // Se não há parâmetros UTM na URL, verificar os cookies
                if (Object.keys(params).length === 0) {
                    var utmParams = ['utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content'];
                    for (var j = 0; j < utmParams.length; j++) {
                        var value = Cookies.get(utmParams[j]);
                        if (value) {
                            params[utmParams[j]] = value;
                        }
                    }
                }
                
                // Se ainda não há parâmetros UTM, usar os padrões (se disponíveis)
                if (Object.keys(params).length === 0 && Object.keys(utmTrkConfig.defaultUtmParams).length > 0) {
                    params = utmTrkConfig.defaultUtmParams;
                }
                
                return params;
            }
            
            // Função para verificar se um link é interno ou externo
            function isExternalLink(url) {
                if (!url || url === '#' || url.startsWith('javascript:') || url.startsWith('mailto:')) {
                    return false;
                }
                
                var urlObj;
                try {
                    urlObj = new URL(url, window.location.origin);
                } catch (e) {
                    return false;
                }
                
                // Verificar se o domínio é o mesmo
                var isExternal = urlObj.hostname !== window.location.hostname;
                
                // Verificar se o domínio está na lista de exclusões
                if (isExternal && utmTrkConfig.excludedDomains.length > 0) {
                    for (var i = 0; i < utmTrkConfig.excludedDomains.length; i++) {
                        if (urlObj.hostname.indexOf(utmTrkConfig.excludedDomains[i]) !== -1) {
                            return false;
                        }
                    }
                }
                
                return isExternal;
            }
            
            // Função para adicionar parâmetros UTM a uma URL
            function addUTMToUrl(url, utmParams) {
                if (!Object.keys(utmParams).length) {
                    return url;
                }
                
                // Ignorar URLs não válidas ou links especiais
                if (!url || url === '#' || url.startsWith('javascript:') || url.startsWith('mailto:')) {
                    return url;
                }
                
                var urlParts = url.split('#');
                var baseUrl = urlParts[0];
                var hash = urlParts.length > 1 ? '#' + urlParts[1] : '';
                
                var separator = baseUrl.indexOf('?') === -1 ? '?' : '&';
                var utmString = '';
                
                for (var key in utmParams) {
                    if (utmParams.hasOwnProperty(key)) {
                        utmString += separator + key + '=' + utmParams[key];
                        separator = '&';
                    }
                }
                
                return baseUrl + utmString + hash;
            }
            
            // Rastreamento de cliques para estatísticas
            function trackClick(element, utmParams) {
                if (!utmTrkConfig.enableAnalytics) {
                    return;
                }
                
                var href = element.getAttribute('href');
                var isExternal = isExternalLink(href);
                
                // Dados para enviar via AJAX
                var data = {
                    action: 'utm_trk_pro_track_click',
                    nonce: utmTrkConfig.nonce,
                    url: href,
                    is_external: isExternal ? 1 : 0,
                    utm_source: utmParams.utm_source || '',
                    utm_medium: utmParams.utm_medium || '',
                    utm_campaign: utmParams.utm_campaign || ''
                };
                
                // Enviar dados via fetch de forma assíncrona (não esperar pela resposta)
                fetch(utmTrkConfig.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams(data)
                }).catch(function(error) {
                    console.error('UTM TRK PRO: Erro ao rastrear clique', error);
                });
            }
            
            // Obter os parâmetros UTM
            var utmParams = getUTMParameters();
            
            try {
                // Função para processar um link
                function processLink(link) {
                    var href = link.getAttribute('href');
                    if (!href) {
                        return;
                    }
                    
                    var isExternal = isExternalLink(href);
                    
                    // Verificar se devemos adicionar UTM a este link com base nas configurações
                    if ((isExternal && utmTrkConfig.trackExternal) || (!isExternal && utmTrkConfig.trackInternal)) {
                        var newHref = addUTMToUrl(href, utmParams);
                        link.setAttribute('href', newHref);
                        
                        // Adicionar evento de clique para rastreamento, se necessário
                        if (utmTrkConfig.enableAnalytics && !link.getAttribute('data-utm-tracked')) {
                            link.setAttribute('data-utm-tracked', 'true');
                            link.addEventListener('click', function() {
                                trackClick(this, utmParams);
                            });
                        }
                    }
                }
                
                // Processar todos os links da página
                var allLinks = document.querySelectorAll('a');
                for (var i = 0; i < allLinks.length; i++) {
                    processLink(allLinks[i]);
                }
                
                // Observar alterações no DOM para aplicar em elementos carregados dinamicamente
                var observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.addedNodes && mutation.addedNodes.length > 0) {
                            for (var i = 0; i < mutation.addedNodes.length; i++) {
                                var node = mutation.addedNodes[i];
                                if (node.nodeType === 1) { // Elemento
                                    var newLinks = node.querySelectorAll('a');
                                    for (var j = 0; j < newLinks.length; j++) {
                                        processLink(newLinks[j]);
                                    }
                                }
                            }
                        }
                    });
                });
                
                observer.observe(document.body, {
                    childList: true,
                    subtree: true
                });
                
            } catch (e) {
                console.error('UTM TRK PRO: Erro ao processar links', e);
            }
        });
        </script>
        <?php
    }
    
    /**
     * Processa o rastreamento de cliques via AJAX
     */
    public function track_click() {
        check_ajax_referer('utm-trk-pro-nonce', 'nonce');
        
        // Verificar se o rastreamento está ativado
        if (!$this->settings->is_tracking_enabled() || !$this->settings->is_analytics_enabled()) {
            wp_send_json_error('Rastreamento desativado');
        }
        
        $url = isset($_POST['url']) ? sanitize_url($_POST['url']) : '';
        $is_external = isset($_POST['is_external']) ? (int) $_POST['is_external'] : 0;
        $utm_source = isset($_POST['utm_source']) ? sanitize_text_field($_POST['utm_source']) : '';
        $utm_medium = isset($_POST['utm_medium']) ? sanitize_text_field($_POST['utm_medium']) : '';
        $utm_campaign = isset($_POST['utm_campaign']) ? sanitize_text_field($_POST['utm_campaign']) : '';
        
        // Registrar clique no log
        $this->log_click($url, $is_external, $utm_source, $utm_medium, $utm_campaign);
        
        wp_send_json_success();
    }
    
    /**
     * Registra um clique no banco de dados
     */
    private function log_click($url, $is_external, $utm_source, $utm_medium, $utm_campaign) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'utm_trk_pro_clicks';
        
        // Verificar se a tabela existe
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
            // A tabela não existe, criar
            $charset_collate = $wpdb->get_charset_collate();
            
            $sql = "CREATE TABLE $table_name (
                id mediumint(9) NOT NULL AUTO_INCREMENT,
                url varchar(500) NOT NULL,
                is_external tinyint(1) NOT NULL,
                utm_source varchar(100) NOT NULL,
                utm_medium varchar(100) NOT NULL,
                utm_campaign varchar(100) NOT NULL,
                click_time datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
                user_id bigint(20) NULL,
                ip_address varchar(100) NULL,
                user_agent varchar(500) NULL,
                PRIMARY KEY  (id)
            ) $charset_collate;";
            
            require_once ABSPATH . 'wp-admin/includes/upgrade.php';
            dbDelta($sql);
        }
        
        // Obter informações adicionais
        $user_id = get_current_user_id();
        $ip_address = sanitize_text_field($_SERVER['REMOTE_ADDR']);
        $user_agent = sanitize_text_field($_SERVER['HTTP_USER_AGENT']);
        
        // Inserir o registro
        $wpdb->insert(
            $table_name,
            array(
                'url' => $url,
                'is_external' => $is_external,
                'utm_source' => $utm_source,
                'utm_medium' => $utm_medium,
                'utm_campaign' => $utm_campaign,
                'user_id' => $user_id,
                'ip_address' => $ip_address,
                'user_agent' => $user_agent
            )
        );
    }
}
